// Header files

  // Classes
  #include "Sensor.h"
  #include "LcdScreen.h"
  #include "VoltageDividerSensor.h"
  #include "CurrentSensor.h"
  #include "InfraredSensor.h"
  #include "MicroSdCardReader.h"
  #include "TorqueTransducerAmpSensor.h"
  #include "ControlConstants.h"


LcdScreen screen( I2C_ADDRESS, COLUMNS, ROWS );

VoltageDividerSensor voltageDividerSensor( VOLTAGE_DIVIDER_ANALOG_PIN, 
                                           VOLTAGE_DIVIDER_HIGH_RESISTOR_KOHM, 
                                           VOLTAGE_DIVIDER_LOW_RESISTOR_KOHM, 
                                           VOLTAGE_UNIT, 
                                           VOLTAGE_DIVIDER_SAMPLE_AMT,
                                           VOLTAGE_DIVIDER_CALIBRATION_CURVE_SLOPE,
                                           VOLTAGE_DIVIDER_CALIBRATION_CURVE_INTERCEPT );

CurrentSensor currentSensor( CURRENT_SENSOR_ANALOG_PIN, 
                             CURRENT_SENSOR_SENSITIVITY, 
                             CURRENT_UNIT, 
                             CURRENT_SENSOR_SAMPLE_AMT,
                             CURRENT_SENSOR_CALIBRATION_CURVE_SLOPE,
                             CURRENT_SENSOR_CALIBRATION_CURVE_INTERCEPT,
                             VOLTAGE_DIVIDER_CALIBRATION_CURVE_SLOPE,
                             VOLTAGE_DIVIDER_CALIBRATION_CURVE_INTERCEPT );

InfraredSensor infraredSensor( INFRARED_SENSOR_OUT_PIN, 
                               ROTATIONAL_SPEED_UNIT, 
                               INFRARED_SENSOR_SAMPLE_AMT ); 

TorqueTransducerAmpSensor torqueTransducerAmpSensor( LOAD_CELL_AMP_DAT_PIN, 
                                                     LOAD_CELL_AMP_CLK_PIN, 
                                                     TORQUE_UNIT, 
                                                     LOAD_CELL_AMP_SAMPLE_AMT,
                                                     TORQUE_TRANSDUCER_CALIBRATION_CURVE_SLOPE,
                                                     TORQUE_TRANSDUCER_CALIBRATION_CURVE_INTERCEPT );

MicroSdCardReader cardReader( SD_CARD_READER_CS_PIN );

Sensor *sensors[ NUM_OF_SENSORS ] = { &voltageDividerSensor,
                                      &currentSensor, 
                                      &infraredSensor, 
                                      &torqueTransducerAmpSensor };


// Prototypes
void checkForMenuInput();
void checkRemainingSampleAmt();
void clearSerialInput();
void collectMeasurements();
int getSerialIntInput();
void handleContinueWritingCode();
void handleStartWritingCode();
void handleStopWritingCode();
void initializeCardReader();
void initializeDataLogging();
void initializeScreen();
void initializeSensors();
void printCardDetectedScreen();
void printCardNotDetectedScreen();
void printInitializingScreen();
void printMeasurements();
void printUnits();
void ( * resetProgram ) ( void ) = 0;
void writeMeasurements();
void writeSensorData();


void checkForMenuInput()
{
  int code;
  
  bool writingToFile = cardReader.getWritingInCard();
  bool cardIsAvailable = cardReader.getCardIsAvailable();
  int numOfWrites = cardReader.getNumOfWrites();

  if ( Serial.available() > 0 )
  {
    code = Serial.parseInt();

    if ( code == START_WRITING && cardIsAvailable && !writingToFile )
    {
      handleStartWritingCode();
    }

    else if ( code == CONTINUE_WRITING && 
              cardIsAvailable && 
              writingToFile &&
              numOfWrites == 0 )
    {
      handleContinueWritingCode();
    }

    else if ( code == STOP_WRITING && cardIsAvailable && writingToFile )
    {
      handleStopWritingCode();
    }

    else if ( code == RESET )
    {
      resetProgram();
    }
  }
}


void checkRemainingSampleAmt()
{
  static bool isPrinted = false;
  int numOfWrites = cardReader.getNumOfWrites();

  if ( numOfWrites == 0 && !isPrinted )
  {
    Serial.println( F( "Done sampling" ) );
    isPrinted = true;
  }

  else if ( numOfWrites > 0 && isPrinted )
  {
    isPrinted = false;
  }
}


void clearSerialInput()
{
  while ( Serial.available() > 0 )
  {
    Serial.read();
  }
}


void collectMeasurements()
{
  unsigned int sensorIdx;

  for ( sensorIdx = 0; sensorIdx < NUM_OF_SENSORS; sensorIdx++ )
  {
    sensors[ sensorIdx ]->measure();
  }
}


int getSerialIntInput()
{
  int retInt;
  
  while ( Serial.available() == 0 );

  retInt = Serial.parseInt();

  clearSerialInput();

  return retInt;
}


void handleContinueWritingCode()
{
  cardReader.resetNumOfWrites();
  Serial.println( F( "Continuing sampling" ) );
}


void handleStartWritingCode()
{
  char fileName[ MAX_FILE_PREFIX_NAME_LEN ] = "LOG";
  int numOfWrites;

  cardReader.setWritingInCard( true );

  cardReader.createFileName( fileName );
  cardReader.openFile();

  screen.clearScreen();

  screen.print( "Writing data to: ", 0 );
  screen.print( cardReader.getFileName(), 1 );
  delay( 2 * ONE_SECOND_MS );
  screen.clearScreen();

  screen.print( "Enter # of samples", 0 );
  screen.print( "NOTE:", 2 );
  screen.print( "-# = continuous DAQ", 3 );

  numOfWrites = getSerialIntInput();
  cardReader.setNumOfWrites( numOfWrites );

  screen.clearScreen();

  cardReader.writeHeader();
  printUnits();
}


void handleStopWritingCode()
{
  cardReader.closeFile();
  cardReader.setWritingInCard( false );
  screen.clearScreen();
  screen.print( "Data written to:", 0 );
  screen.print( cardReader.getFileName(), 1 );
  delay( 2 * ONE_SECOND_MS );
  screen.clearScreen();
  printUnits();
}


void initializeCardReader()
{
  bool cardReaderIsInitialized = cardReader.initialize();

  screen.clearScreen();

  if ( cardReaderIsInitialized )
  {
    printCardDetectedScreen();
  }

  else
  {
    printCardNotDetectedScreen();
  }

  delay( 2 * ONE_SECOND_MS );
  screen.clearScreen();
}


void initializeDataLogging()
{
  initializeScreen();
  initializeCardReader();
  printUnits();
}


void initializeScreen()
{
  screen.initialize();
  printInitializingScreen();
}


void initializeSensors()
{
  unsigned int sensorIdx;

  for ( sensorIdx = 0; sensorIdx < NUM_OF_SENSORS; sensorIdx++ )
  {
    sensors[ sensorIdx ]->initialize();
  }
}


void printCardDetectedScreen()
{
  screen.print( "SD Card Detected", 0 );
}


void printCardNotDetectedScreen()
{
  screen.print( "SD Card Not Detected", 0 );
}


void printInitializingScreen()
{
  screen.print( "CWC GEN Fall '25", 0 );
  screen.print( "Initializing card", 2 );
  screen.print( "reader and sensors", 3 );
}


void printMeasurements()
{
  unsigned int sensorIdx;
  Sensor *sensor;
  float measurement;
  char *unit;

  for ( sensorIdx = 0; sensorIdx < NUM_OF_SENSORS; sensorIdx++ )
  {
    sensor = sensors[ sensorIdx ];
    measurement = sensor->getMeasurement();
    unit = sensor->getUnit();

    screen.printMeasurement( measurement, unit, sensorIdx );
  }
}


void printUnits()
{
  unsigned int sensorIdx;
  Sensor *sensor;
  char *unit;

  for ( sensorIdx = 0; sensorIdx < NUM_OF_SENSORS; sensorIdx++ )
  {
    sensor = sensors[ sensorIdx ];
    unit = sensor->getUnit();

    screen.printUnit( unit, sensorIdx );
  }
}


void writeMeasurements()
{
  cardReader.writeTime();
  writeSensorData();
  cardReader.writeNewLine();
}


void writeSensorData()
{
  unsigned int sensorIdx;

  for ( sensorIdx = 0; sensorIdx < NUM_OF_SENSORS; sensorIdx++ )
  {
    cardReader.writeData( sensors[ sensorIdx ] );
  }
}


void setup()
{
  Serial.begin( BAUD_RATE );
  initializeDataLogging();
  initializeSensors();
}


void loop()
{
  checkForMenuInput();
  checkRemainingSampleAmt();
  collectMeasurements();
  printMeasurements();
  writeMeasurements();
}
