/*************************************************************************
 *
 *    File Name: docwho.c
 *
 *    Description: Main code for Doc Who 
 *
 *    History:
 *       20 Mar 2000: Created; Joshua Colvin
 *       30 Mar 2000: Added comment headers; Jason Campbell
 *       01 Apr 2000: Filled in comment headers; Joshua Colvin
 *
 *************************************************************************
 * Copyright (C) 2000 Jason Campbell, Joshua Colvin, Jason Sherrill,
 *                    Ben Tobin
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, In., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 *************************************************************************/

#include <PalmOS.h>

#include "docwhoRsc.h"

#include "doclist.h"
#include "doclistdb.h"
#include "docview.h"
#include "docdetails.h"
#include "version.h"


/*************************************************************************
 *
 *    Function Name: ApplicationHandleEvent
 *
 *    Purpose:       assign appropriate event handlers when form changes
 *
 *    Parameters:    eventP: pointer to event to process
 *
 *    Return Value:  true only if event handled
 *
 *
 *************************************************************************/
static Boolean ApplicationHandleEvent(EventPtr eventP)
{
	FormPtr frm;
	UInt16    formId;
	Boolean handled = false;
	
	if (eventP->eType == frmLoadEvent)
	{
		//Initialize and load requested form
		formId = eventP->data.frmLoad.formID;
		frm = FrmInitForm(formId);
		FrmSetActiveForm(frm);
		
		// Set event handler if necessary
		switch(formId)
		{
			case DocListForm:
				FrmSetEventHandler(frm, DocListFormHandleEvent);
				break;
			case DocViewForm:
				FrmSetEventHandler(frm, DocViewFormHandleEvent);
				break;
			case DocDetailsForm:
				FrmSetEventHandler(frm, DocDetailsFormHandleEvent);
				break;
		}
		
		handled = true;
	}
	
	return handled;
}

/*************************************************************************
 *
 *    Function Name: StartApplicaton
 *
 *    Purpose:       Get preferences, open (or create) app database 
 *
 *    Parameters:    none
 *
 *    Return Value:  0 if no errors
 *
 *
 *************************************************************************/
static Err StartApplication(void)
{
	// Initialize database for keeping track of Doc file info
	DocListDBInit();

	// We do not need to register with exchange manager because the
	//  the system already handles receiving .prc files
	
	FrmGotoForm(DocListForm);
	
	return errNone;
}

/*************************************************************************
 *
 *    Function Name: StopApplication
 *
 *    Purpose:       Save preferences, close forms, close app database 
 *
 *    Parameters:    none
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void StopApplication(void)
{
	FrmSaveAllForms();
	FrmCloseAllForms();
	
	// Close Doc file info database
	DocListDBClose();
}

/*************************************************************************
 *
 *    Function Name: EventLoop
 *
 *    Purpose:       The main event loop 
 *
 *    Parameters:    none
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void EventLoop(void)
{
	Err err;
	EventType eventP;
	
	do
	{
		EvtGetEvent(&eventP, evtWaitForever);
		if (! SysHandleEvent (&eventP))
			if (! MenuHandleEvent (NULL, &eventP, &err))
				if (! ApplicationHandleEvent (&eventP))
					FrmDispatchEvent (&eventP);
	} while (eventP.eType != appStopEvent);
}

/*************************************************************************
 *
 *    Function Name: PilotMain
 *
 *    Purpose:       Main entry point; it is unlikely you will need to 
 *                   change this except to handle other launch command codes 
 *
 *    Parameters:    cmd:         Launch code
 *                   cmdPBP:      Pointer to structure associated with cmd
 *                   launchFlags: Extra information about launch
 *
 *    Return Value:  errNone if no errors
 *
 *
 *************************************************************************/
UInt32 PilotMain(UInt16 cmd, MemPtr cmdPBP, UInt16 launchFlags)
{
	Err err;
	Boolean result;
	
	result = VersionCheckOS(RomVer2_0);
	if (false == result)
	{
		// ROM version is too old
		//#### Should print error message before exiting program
		return sysErrRomIncompatible;
	}
	else if (sysAppLaunchCmdNormalLaunch == cmd)
	{
		err = StartApplication();
		if (err)
		{
			return err;
		}
		EventLoop();
		StopApplication();
	}
	
	return errNone;
}
