/*************************************************************************
 *
 *  File Name: docview.c
 *
 *  Description: All functions pertaining to DocViewForm
 *
 *  History:
 *     20 Mar 2000: Created; Joshua Colvin
 *     30 Mar 2000: Added comment headers; Jason Campbell
 *     01 Apr 2000: Filled in comment headers; Joshua Colvin
 *     05 Apr 2000: Implemented DocViewSetDocInfo; Jason Sherrill
 *     05 Apr 2000: Uncommented calls to DocOpen & TextFieldFromHandle; Jason Sherrill
 *     18 Apr 2000: Added call to DocClose in frmCloseEvent; Jason Sherrill
 *     20 Apr 2000: Created DocViewTrackScroll; Jason Sherrill
 *     22 Apr 2000: Created DocViewSetDirty & setClean; Jason Sherrill
 *
 *************************************************************************
 * Copyright (C) 2000 Jason Campbell, Joshua Colvin, Jason Sherrill,
 *                    Ben Tobin
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, In., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 *************************************************************************/

#include <PalmOS.h>
#include "docwhoRsc.h"

#include "docview.h"
#include "doc.h"
#include "textfield.h"
#include "control.h"
#include "menu.h"

// Globals to keep track of Doc file to edit
UInt16 DocCardNumber;
LocalID DocDBID;
MemHandle textH;

Boolean nextDirty;

static void DocViewTrackScroll(WinDirectionType d);


/*************************************************************************
 *
 *    Function Name: DocViewFormHandleEvent
 *
 *    Purpose:       Handle events for DocViewForm
 *
 *    Parameters:    eventP: pointer to event to process
 *
 *    Return Value:  true only if event handled
 *
 *
 *************************************************************************/

Boolean DocViewFormHandleEvent (EventPtr eventP)
{
	Boolean handled = false;
	Boolean edited;
	FormPtr frmP;
	FieldPtr fieldP;

	switch (eventP->eType) {
		case frmOpenEvent:
			frmP = FrmGetActiveForm();

			textH = DocOpen(DocCardNumber, DocDBID);
			TextFieldFromHandle(DocField, textH);
			nextDirty = false;

			FrmSetFocus(frmP, FrmGetObjectIndex(frmP, DocField));
			FrmDrawForm(frmP);
			handled = true;
			break;
			
		case frmCloseEvent:
			frmP = FrmGetActiveForm();
			fieldP = ControlGetPtrByID(frmP, DocField) ;

			edited = FldDirty(fieldP);
			// Pass <edited> instead of false to allow saving
			DocClose(false);
			handled = true;
			break;
			
		case menuEvent:
			handled = MenuDocView(eventP);
			break;
		
		case keyDownEvent:
			// Scroll if hardware up/down key pressed	
			if (eventP->data.keyDown.chr == pageUpChr ||
				 eventP->data.keyDown.chr == pageDownChr)
			{
				WinDirectionType d;
				
				frmP = FrmGetActiveForm();
				fieldP = ControlGetPtrByID(frmP, DocField) ;
				if (eventP->data.keyDown.chr == pageUpChr)
				{
					d = winUp;
				}
				else
				{
					d = winDown;
				}
				//#### Must write scroll handlers for field
				FldScrollField(fieldP, 1, d);

				DocViewTrackScroll(d);

				handled = true;
			}

			break;
		
		default:
			break;
	}

	return handled;
}


void DocViewSetDocInfo(UInt16 cardNum, LocalID dbID)
{
    DocCardNumber = cardNum;
    DocDBID = dbID;
}


/*************************************************************************
 *
 *    Function Name: DocViewTrackScroll
 *
 *    Purpose:       Tracks the scroll position in the text field and if it
 *                   had been edited. Calls DocMasterScroll at boundries
 *
 *    Parameters:    d: The direction of scrolling
 *
 *    Return Value:  nothing
 *
 *
 *************************************************************************/

static void DocViewTrackScroll(WinDirectionType d)
{
	FormPtr   frmPtr;
	FieldPtr  fldPtr;

	MemHandle oldTxtHdl;
	MemHandle newTxtHdl;

	Boolean   edited;
	Boolean   keepSame;
	UInt16    lenChanged;
	UInt16    scrlPos;
	UInt16    lnScrlPos;
	UInt16    numLines;
	UInt16    numLnPerPg;
	UInt16    transLineUp;
	UInt16    transLineDn;

	frmPtr = FrmGetActiveForm();
	fldPtr = ControlGetPtrByID(frmPtr, DocField);

	FldGetScrollValues(fldPtr, &lnScrlPos, &numLines, &numLnPerPg);

	transLineUp = 1;					// 1st line
	transLineDn = numLines - numLnPerPg - 1;	// top line on last page

	if (d == winUp)
	{
		if (lnScrlPos >= transLineUp)
			return;
	}
	else if (d == winDown)
	{
		if (lnScrlPos <= transLineDn)
			return;
	}
	else
	{
		ErrDisplay("Error: Doc Scroll Direction Wrong");
		return;
	}

	// We've reached a boundry of the text field!
	scrlPos = FldGetScrollPosition(fldPtr);
	edited = FldDirty(fldPtr);
	oldTxtHdl = FldGetTextHandle(fldPtr);
	keepSame = false;

	// Pass <edited> instead of false to allow saving
	newTxtHdl = DocMasterScroll(d, false, &lenChanged, &keepSame);

	if (keepSame)
		return;

// When newTxtHdl actually pointed to the new chunk this was still true.
//	if (newTxtHdl == NULL) This was the big bug I've been hunting for for so long!!!
//		newTxtHdl = oldTxtHdl;

	FldSetTextHandle(fldPtr, newTxtHdl);

	FldDrawField(fldPtr);

	if (d == winUp)
		FldSetScrollPosition(fldPtr, scrlPos+lenChanged);
	else  // down
		FldSetScrollPosition(fldPtr, scrlPos-lenChanged);

	if (oldTxtHdl)
		MemHandleFree(oldTxtHdl);

	if (nextDirty)
		FldSetDirty(fldPtr, true);
	else
		FldSetDirty(fldPtr, false);

	FrmSetFocus(frmPtr, FrmGetObjectIndex(frmPtr, DocField));
	FrmDrawForm(frmPtr);
}

void DocViewSetDirty(void)
{
	nextDirty = true;
}

void DocViewSetClean(void)
{
	nextDirty = false;
}
