/*************************************************************************
 *
 *    File Name:   docutil.h
 *
 *    Description: Utilities for working with Doc files
 *
 *    History:
 *       27 Apr 2000: Created; Joshua Colvin
 *
 *************************************************************************
 * Copyright (C) 2000 Jason Campbell, Joshua Colvin, Jason Sherrill,
 *                    Ben Tobin
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, In., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 *************************************************************************/

#include <PalmOS.h>
#include "docutil.h"
#include "docwhoRsc.h"
#include "database.h"
#include "doclistdb.h"

// Local function prototypes
static Err DocUtilBlankDB(UInt16 *cardNoP, LocalID *dbIDP, Char *name);

/*************************************************************************
 *
 *    Function Name: DocUtilBlankDB
 *
 *    Purpose:       Create a blank DB of type Doc with a unique name
 *
 *    Parameters:    In:
 *                   name:    name to start with
 *
 *                   Out:
 *                   cardNoP: card number of blank Doc
 *                   dbIDP:   DB ID of blank Doc
 *
 *    Return Value:  non-zero if error occurs
 *
 *
 *************************************************************************/
static Err DocUtilBlankDB(UInt16 *cardNoP, LocalID *dbIDP, Char *name)
{
	// tempName* and nameCounter variables are for creating unique names
	Char tempName[dmDBNameLength] = NewDocName;
	Char tempNameNumber[dmDBNameLength];
	Int8 nameCounter;
	LocalID dbID;
	Err err;
	
	// Initialize variables
	nameCounter = 1;
	err = 0;
	StrNCopy(tempName, name, dmDBNameLength);

	// Check if name already exists
	dbID = DmFindDatabase(DefaultCardNo, tempName);
	
	// If name already exists, try new names until found unique one
	//  or more than 100 different names tried
	while ( (0 != dbID) && (nameCounter < 100) )
	{
		// Initialize tempName, leave space for 2 digit number
		StrNCopy(tempName, name, dmDBNameLength - 2);
		
		// Add a number onto the end of the name
		nameCounter++;
		StrIToA(tempNameNumber, nameCounter);
		StrNCat(tempName, tempNameNumber, dmDBNameLength);
		dbID = DmFindDatabase(DefaultCardNo, tempName);
	}
	// Set cardNoP
	*cardNoP = DefaultCardNo;
	// Try creating database
	err = DmCreateDatabase(*cardNoP, tempName, DocCreator, DocType,
		                       false);
	// If no errors, get dbID
	if (0 == err)
	{
		*dbIDP = DmFindDatabase(*cardNoP, tempName);
		
		// Check if error occured
		if (0 == *dbIDP)
		{
			err = DmGetLastErr();
		}
	}
	return err;
}

/*************************************************************************
 *
 *    Function Name: DocUtilNew
 *
 *    Purpose:       Create an empty Doc file
 *
 *    Parameters:    none
 *
 *    Return Value:  non-zero if error occurs
 *
 *
 *************************************************************************/
Err DocUtilNew(void)
{
	Err err;
	UInt16 cardNo;
	LocalID dbID;
	DmOpenRef dbP;
	DocRec0Type recZeroInit;
	MemHandle recH;
	MemPtr recP;
	UInt16 index;
	
	// Create new DB with unique name
	err = DocUtilBlankDB(&cardNo, &dbID, NewDocName);
	if (err)
	{
		return err;
	}
	
	// Open new DB
	dbP = DmOpenDatabase(cardNo, dbID, dmModeWrite);
	if (NULL == dbP)
	{
		err = DmGetLastErr();
		return err;
	}
	
	// Create record 0
	recH = DmNewRecord(dbP, &index, sizeof(DocRec0Type));
	if (NULL == recH)
	{
		err = DmGetLastErr();
		return err;
	}
	
	// If new record is not record 0, something is wrong
	ErrNonFatalDisplayIf(0 != index, "recZero isn't at index zero")
	
	// Lock record 0
	recP = MemHandleLock(recH);
	if (NULL == recP)
	{
		err = DmGetLastErr();
		return err;
	}
	
	// Fill in record 0
	recZeroInit.version = 2;
	recZeroInit.spare   = 0;
	recZeroInit.length  = 0;
	recZeroInit.numRecs = 5;
	recZeroInit.recSize = DefaultDocRecordSize;
	recZeroInit.position= 0;
	recZeroInit.sizes   = NULL;
	
	// Copy into DB record
	DmWrite(recP, 0, &recZeroInit, sizeof(DocRec0Type));
	
	// Unlock record 0
	err = MemHandleUnlock(recH);
	if (err)
	{
		return err;
	}
	
	// Release record 0
	err = DmReleaseRecord(dbP, index, true);
	if (err)
	{
		return err;
	}	

	// Create initial blank record for Doc file
	index = 1;
	recH = DmNewRecord(dbP, &index, DefaultDocRecordSize);
	if (NULL == recH)
	{
		err = DmGetLastErr();
		return err;
	}
	
	// If new record is not record 1, something is wrong
	ErrNonFatalDisplayIf(1 != index, "recOne isn't at index one")
	
	// Lock record 1
	recP = MemHandleLock(recH);
	if (NULL == recP)
	{
		err = DmGetLastErr();
		return err;
	}
	
	// Initialize record 1
	DmSet(recP, 0, DefaultDocRecordSize, 0);

	// Unlock record 1
	err = MemHandleUnlock(recH);
	if (err)
	{
		return err;
	}	

	// Release record 1
	err = DmReleaseRecord(dbP, index, true);
	if (err)
	{
		return err;
	}	
	
	// Close new DB
	err = DmCloseDatabase(dbP);	

	return err;	
}

/*************************************************************************
 *
 *    Function Name: DocUtilDuplicate
 *
 *    Purpose:       Create a duplicate of a Doc file
 *
 *    Parameters:    recordIndex: Index in DocListDB for Doc file info
 *
 *    Return Value:  non-zero if error occurs
 *
 *
 *************************************************************************/
Err DocUtilDuplicate(UInt16 recordIndex)
{
	UInt16 cardNo;
	LocalID dbID;
	Char name[dmDBNameLength];
	Err err;
	
	DocListDBGetInfo(recordIndex, &cardNo, &dbID, name, NULL,
	                 NULL, NULL, NULL);
	
	// #### finish
	return err;
}
