/*************************************************************************
 *
 *    File Name:   doclistdb.c
 *
 *    Description: Functions for maintaining database for storing info
 *                  displayed by the Doc List
 *
 *    History:
 *       20 Mar 2000: Created; Joshua Colvin
 *       30 Mar 2000: Added comment headers; Jason Campbell
 *       01 Apr 2000: Filled in comment headers; Joshua Colvin
 *       04 Apr 2000: DB wasn't being closed in DBSetCategory,
 *                     fixed; Joshua Colvin
 *       13 Apr 2000: Added DocListDBInitAppInfo; Joshua Colvin
 *       14 Apr 2000: Added DocListDBCategorySelect; Joshua Colvin
 *       15 Apr 2000: Finished DocListDB version checking; Joshua Colvin
 *       18 Apr 2000: Removed saving App Info block of each
 *                     Doc file; Joshua Colvin
 *       28 Apr 2000: Added DocListDBUpdateCategory; Joshua Colvin
 *
 *************************************************************************
 * Copyright (C) 2000 Jason Campbell, Joshua Colvin, Jason Sherrill,
 *                    Ben Tobin
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, In., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 *************************************************************************/

#include <PalmOS.h>

#include "doclistdb.h"
#include "docwhoRsc.h"
#include "control.h"
#include "version.h"
#include "database.h"

// local Globals
static DmOpenRef gDocListDBRef;
static Err gErr;

// Private function prototypes
static void DocListDBRemoveAllRecords();
static void DocListDBPackRecord(const DocInfoType *infoP,
                                MemHandle recordH);
static void DocListDBCreateRecord(DocInfoType *infoP);
static void DocListDBInitAppInfo();
static void DocListDBDelete();
static void DocListDBSetCategory(UInt16 recordIndex);

/*************************************************************************
 *
 *    Function Name: DocListDBInitAppInfo
 *
 *    Purpose:       Create AppInfo block if needed, and loads the default
 *                    category strings
 *
 *    Parameters:    none
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void DocListDBInitAppInfo()
{
	LocalID dbID;
	UInt16 cardNo;
	LocalID appInfoID;
	AppInfoPtr appInfoP;
	MemHandle appInfoH;

	// Get info needed to get AppInfo block
	gErr = DmOpenDatabaseInfo(gDocListDBRef, &dbID, NULL, NULL,
	                          &cardNo, NULL);
	ErrNonFatalDisplayIf(gErr,
	                     "Could not get database info in InitAppInfo");
	
	// Get AppInfo block
	gErr = DmDatabaseInfo(cardNo, dbID, NULL, NULL, NULL, NULL, NULL,
	                       NULL, NULL, &appInfoID, NULL, NULL, NULL);
	ErrNonFatalDisplayIf(gErr,
	                     "Could not get AppInfo info in InitAppInfo");
	

	// Create a new AppInfo block if needed
	if (NULL == appInfoID)
	{
		appInfoH = DmNewHandle (gDocListDBRef, sizeof(AppInfoType));
		ErrNonFatalDisplayIf(NULL == appInfoH,
		                     "Could not create new handle");
		
		appInfoID = MemHandleToLocalID (appInfoH);
		
		gErr = DmSetDatabaseInfo(cardNo, dbID, NULL, NULL, NULL, NULL,
		                 NULL, NULL, NULL, &appInfoID, NULL, NULL, NULL);
		ErrNonFatalDisplayIf(gErr, "Could not set database info.");
	}


	// Get pointer to appInfo and initialize with our default categories	
	appInfoP = MemLocalIDToLockedPtr(appInfoID, cardNo);

	ErrNonFatalDisplayIf(NULL == appInfoP,
	                     "couldn't get pointer to AppInfo block");
	
	// Clear the app info block
	DmSet(appInfoP, 0, sizeof(AppInfoType), 0);

	// Initialize the categories
	CategoryInitialize(appInfoP, DefaultCategoryList);

	MemPtrUnlock(appInfoP);
}

/*************************************************************************
 *
 *    Function Name: DocListDBRemoveAllRecords
 *
 *    Purpose:       Remove all records in DocListDB
 *
 *    Parameters:    none
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void DocListDBRemoveAllRecords()
{
	UInt16 index;
	MemHandle recordH;

	index = 0;
	while (NULL != (recordH = DmQueryNextInCategory(gDocListDBRef, 
	       &index, dmAllCategories)))
	{
		//####		
		//
		//FrmCustomAlert(DebugAlert, "remove record", NULL, NULL);
	
		gErr = DmRemoveRecord(gDocListDBRef, index);
		ErrFatalDisplayIf(gErr, "problem deleting records from DocListDB");
	}
}

/*************************************************************************
 *
 *    Function Name: DocListDBDelete
 *
 *    Purpose:       Delete DocListDB
 *
 *    Parameters:    none
 *
 *    Return Value:  none
 *
 *    Global Vars:   gDocListDBRef: Sets to NULL
 *
 *
 *************************************************************************/
static void DocListDBDelete()
{
	LocalID dbID;
	UInt16 dbCardNo;
	
	// Get info on database
	DmOpenDatabaseInfo(gDocListDBRef, &dbID, NULL, NULL,
	                   &dbCardNo, NULL); 
	
	// Close old doc version
	gErr = DmCloseDatabase(gDocListDBRef);
	ErrFatalDisplayIf(gErr, "Unable to close DocListDB");
	// Delete old doc version
	gErr = DmDeleteDatabase(dbCardNo, dbID);
	ErrFatalDisplayIf(gErr, "Unable to delete DocListDB");
	
	// reset gDocListDBRef since it is no longer valid
	gDocListDBRef = NULL;
}

/*************************************************************************
 *
 *    Function Name: DocListDBInit
 *
 *    Purpose:       Open DocListDB, creating if necessary
 *
 *    Parameters:    none
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
void DocListDBInit(void)
{
	Boolean result;
	
	// Try to open DocListDB
	gDocListDBRef = DmOpenDatabaseByTypeCreator(DocDBType,
	                                            DocWhoCreatorID,
	                                            dmModeReadWrite);
	
	// If opened DocListDB, check version number
	if (gDocListDBRef)
	{
		result = VersionCheckOpenDB(gDocListDBRef, CurrentDocListDBVersion);
				
		// If version does not match, remove old DocListDB
		if (false == result)
		{
			// If version does not match, remove old DocListDB
			//  (gDocListDBRef will be reset to NULL)
			DocListDBDelete();
			
		} // (docListDBVersion < CurrentDocListDBVersion)
	} // (gDocListDBRef)
		
	if (NULL == gDocListDBRef)
	{
		// If DocListDB not found, create a new one
		gErr = DmCreateDatabase(DefaultCardNo, DocDBName,
		                        DocWhoCreatorID, DocDBType, false);
		ErrFatalDisplayIf(gErr, "DmCreateDatabase error in DBInit");
		
		// Open new DocListDB
		gDocListDBRef = DmOpenDatabaseByTypeCreator(DocDBType,
	                                                   DocWhoCreatorID,
	                                                   dmModeReadWrite);

		// If still not opened, something is very wrong		
		if (NULL == gDocListDBRef)
		{
			gErr = DmGetLastErr();
			ErrDisplay("Unable to open DocListDB");
		} // NULL == gDocDBOpenRef (second try)
		
		// Set version number of new DocListDB
		VersionSetOpenDB(gDocListDBRef, CurrentDocListDBVersion);
		
	} // NULL == gDocDBOpenRef (first try)
	
	// Create AppInfo block with default category names if it does not exist
	DocListDBInitAppInfo();

}

/*************************************************************************
 *
 *    Function Name: DocListDBClose
 *
 *    Purpose:       Close DocListDB
 *
 *    Parameters:    none
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
void DocListDBClose(void)
{
	// Keep database around for keeping track of category names
	//  but get rid of records to save space, they are aways regenerated
	DocListDBRemoveAllRecords();
	DmCloseDatabase(gDocListDBRef);
}

/*************************************************************************
 *
 *    Function Name: DocListDBPackRecord
 *
 *    Purpose:       resize recordH and copy all info in infoP into recordH
 *
 *    Parameters:    infoP:   pointer to information about Doc file
 *                   recordH: unlocked handle of record entry in DocListDB
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void DocListDBPackRecord(const DocInfoType *infoP,
                                MemHandle recordH)
{
	UInt32 length;
	MemPtr recordP;
	
	length = sizeof(infoP->cardNo) +
	         sizeof(infoP->dbID) +
	         sizeof(infoP->attributes) +
	         sizeof(infoP->crDate) +
	         sizeof(infoP->modDate) +
	         sizeof(infoP->bckUpDate) +
	         StrLen(infoP->name) + 1;

	// Resize handle
	gErr = MemHandleResize(recordH, length);
	ErrFatalDisplayIf(gErr,
	      "Unable to resize docInfoDBEntry in DocListDBPackRecord");

	// Lock record
	recordP = MemHandleLock(recordH);
	ErrFatalDisplayIf(!recordP,
	                  "unable to lock recordH in DocListDBPackRecord");

	// Copy the fields
	DmWrite(recordP, OffsetOf(PackedDocInfoType, cardNo),
	        &infoP->cardNo, sizeof(infoP->cardNo));
	DmWrite(recordP, OffsetOf(PackedDocInfoType, dbID),
	        &infoP->dbID, sizeof(infoP->dbID));
	DmWrite(recordP, OffsetOf(PackedDocInfoType, attributes),
	        &infoP->attributes, sizeof(infoP->attributes));
	DmWrite(recordP, OffsetOf(PackedDocInfoType, crDate),
	        &infoP->crDate, sizeof(infoP->crDate));
	DmWrite(recordP, OffsetOf(PackedDocInfoType, modDate),
	        &infoP->modDate, sizeof(infoP->modDate));
	DmWrite(recordP, OffsetOf(PackedDocInfoType, bckUpDate),
	        &infoP->bckUpDate, sizeof(infoP->bckUpDate));
	DmStrCopy(recordP, OffsetOf(PackedDocInfoType, name),
	          infoP->name);
	
	// Unlock record
	MemHandleUnlock(recordH);
}

/*************************************************************************
 *
 *    Function Name: DocListDBSetCategory
 *
 *    Purpose:       Get category of Doc file referenced by index and set
 *                   category of record specified by index
 *
 *    Parameters:    index: index of record to set category
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void DocListDBSetCategory(UInt16 recordIndex)
{
	UInt16 cardNo;
	LocalID dbID;
	UInt16 docAttr;
	UInt16 recAttr;
	DmOpenRef docR;
	
	// Get Doc info from DocDB
	DocListDBGetInfo(recordIndex, &cardNo, &dbID, NULL, NULL, NULL, NULL,
	                 NULL);
	                 
	// Open doc file and get attribute of record 0
	docR = DmOpenDatabase(cardNo, dbID,
	                         dmModeReadOnly);
	DmRecordInfo(docR, 0, &docAttr, NULL, NULL);
	
	// Get DB record attribute
	DmRecordInfo(gDocListDBRef, recordIndex, &recAttr, NULL, NULL);
	
	// Keep non-category parts of DB record attributes
	recAttr &= ~dmRecAttrCategoryMask;
	
	// Ignore non-category parts of doc attributes
	docAttr &= dmRecAttrCategoryMask;
	
	// Combine category and non-category attributes
	recAttr |= docAttr;
	
	// Set DB record attributes
	DmSetRecordInfo(gDocListDBRef, recordIndex, &recAttr, NULL);
	
	DmCloseDatabase(docR);
}
	
/*************************************************************************
 *
 *    Function Name: DocListDBCreateRecord
 *
 *    Purpose:       Create a new record in DocListDB
 *
 *    Parameters:    infoP: information to initialize new record with
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void DocListDBCreateRecord(DocInfoType *infoP)
{
	UInt16 recordIndex;
	MemHandle recordH;

	// Create new record at end of database
	recordIndex = dmMaxRecordIndex;
	recordH = DmNewRecord(gDocListDBRef, &recordIndex,
	                             20);
	ErrFatalDisplayIf(!recordH,
	             "Unable to get handle for new record in DocListDB");
			
	// Copy the fields into the record
	DocListDBPackRecord(infoP, recordH);

	// Release recordH,
	DmReleaseRecord(gDocListDBRef, recordIndex, true);
	DocListDBSetCategory(recordIndex);
}

/*************************************************************************
 *
 *    Function Name: DocListDBPopulate
 *
 *    Purpose:       Create a new record for each Doc file found
 *
 *    Parameters:    none
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
void DocListDBPopulate(void)
{
	DmSearchStateType stateInfo;
	DocInfoType info;
	Char nameStr[dmDBNameLength];
	
	// First remove all old records
	DocListDBRemoveAllRecords();

	// Get first Doc file
	gErr = DmGetNextDatabaseByTypeCreator(true, &stateInfo,
	           DocType, DocCreator, false, &info.cardNo, &info.dbID);
	
	while (!gErr && info.dbID)
	{
		// Get information on Doc file
		gErr = DmDatabaseInfo(info.cardNo,
		                      info.dbID,
		                      nameStr,
		                      &info.attributes,
		                      NULL, // Don't retrieve version
		                      &info.crDate,
		                      &info.modDate,
		                      &info.bckUpDate,
		                      NULL, // Don't retrieve modification number
		                      NULL, // Don't retrieve App Info block
		                      NULL, // Don't retrieve sort info
		                      NULL, // Don't retrieve type (already known)
		                      NULL  // Don't retrieve creator (already known)
		                     );

		// Point info.name to location of name string
		info.name = nameStr;
		
		// Store information on Doc file
		DocListDBCreateRecord(&info);
		
		// Now get the next Doc file
		gErr = DmGetNextDatabaseByTypeCreator(false, &stateInfo,
		           DocType, DocCreator, false, &info.cardNo, &info.dbID);
	} // Loop until there are no more doc files, or an error occured
}

/*************************************************************************
 *
 *    Function Name: DocListDBGetCategory
 *
 *    Purpose:       Return the category of a record
 *
 *    Parameters:    recordIndex: index of record
 *
 *    Return Value:  Category of record
 *
 *
 *************************************************************************/
UInt16 DocListDBGetCategory(UInt16 recordIndex)
{
	UInt16 attr;

	// Get record attributes
	DmRecordInfo(gDocListDBRef, recordIndex, &attr, NULL, NULL);
	
	// Remove all attributes except category
	attr &= dmRecAttrCategoryMask;
	
	return attr;
}		

/*************************************************************************
 *
 *    Function Name: DocListDBGetInfo
 *
 *    Purpose:       Get information about a Doc file
 *
 *    Parameters:    In:
 *                   recordIndex: Index of record containing Doc file info
 *                   name:        Must point to Char array of dmDBNameLength
 *                   Out:
 *                   cardNoP: Pointer to card number Doc file stored in
 *                   dbIDP:       Pointer to localID of Doc file
 *                   attributesP: Pointer to attributes
 *                   crDateP:     Pointer to creation date
 *                   modDateP:    Pointer to modification date
 *                   bckUpDateP:  Pointer to last backup date
 *                   name:        Pointer to name string
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
void DocListDBGetInfo(UInt16 recordIndex,
                      UInt16 *cardNoP,
                      LocalID *dbIDP,
                      Char *name,
                      UInt16 *attributesP,
                      UInt32 *crDateP,
                      UInt32 *modDateP,
                      UInt32 *bckUpDateP)
{
	MemHandle recordH;
	PackedDocInfoType *recordP;
	
	// Get DB entry
	recordH = DmGetRecord(gDocListDBRef, recordIndex);
	recordP = MemHandleLock(recordH);
	
	// Return selected value(s)
	if (cardNoP)
		*cardNoP = recordP->cardNo;
	if (dbIDP)
		*dbIDP = recordP->dbID;
	if (name)
		MemMove(name, recordP->name, (StrLen(recordP->name)+1));
	if (attributesP)
		*attributesP = recordP->attributes;
	if (crDateP)
		*crDateP = recordP->crDate;
	if (modDateP)
		*modDateP = recordP->modDate;
	if (bckUpDateP)
		*bckUpDateP = recordP->bckUpDate;

	// Release record
	MemHandleUnlock(recordH);
	DmReleaseRecord(gDocListDBRef, recordIndex, false);
}

/*************************************************************************
 *
 *    Function Name: DocListDBGetNameHandle
 *
 *    Purpose:       Setup info needed to put name part of record into
 *                    field
 *
 *    Parameters:    In:
 *                   recordIndex: index of record to use
 *                   Out:
 *                   dataHandleP: Pointer to unlocked handle of record
 *                   dataOffsetP: Pointer to offset to beginning of name
 *                   dataSizeP:   Pointer to allocated size of string
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
void DocListDBGetNameHandle(UInt16 recordIndex, MemHandle *dataHandleP,
                            Int16 *dataOffsetP, Int16 *dataSizeP)
{
	PackedDocInfoType *dataP;
	
	*dataHandleP = DmQueryRecord(gDocListDBRef, recordIndex);
	ErrFatalDisplayIf(!dataHandleP, "couldn't get record in GetNameHandle");
	
	// Set offset to beginning of name
	*dataOffsetP = OffsetOf(PackedDocInfoType, name);
	
	// Lock record to measure string size
	dataP = MemHandleLock(*dataHandleP);
	ErrFatalDisplayIf(!dataP, "couldn't lock handle in GetNameHandle");
	
	// Allocated size for string is same as string length + 1
	//  (for terminator)
	*dataSizeP = StrLen(dataP->name) + 1;
	
	// Unlock record
	gErr = MemHandleUnlock(*dataHandleP);
	ErrFatalDisplayIf(gErr,
	                  "Unable to unlock dataHandle in DBGetNameHandle");
}

/*************************************************************************
 *
 *    Function Name: DocListDBUpdateDocName
 *
 *    Purpose:       Update Doc name pointed to by recordIndex to match
 *                    name contained in DocListDB->recordIndex
 *
 *    Parameters:    recordIndex: index of record
 *
 *    Return Value:  true if table needs to be redrawn
 *
 *
 *************************************************************************/
Boolean DocListDBUpdateDocName(UInt16 recordIndex)
{
	UInt16 cardNo;
	LocalID dbID;
	Char name[dmDBNameLength];
	
	// Get Doc info from DocDB
	DocListDBGetInfo(recordIndex, &cardNo, &dbID, name, NULL, NULL, NULL,
	                 NULL);
	                 
	
	// #### Do we want to allow user to confirm name change?
	
	DmSetDatabaseInfo(cardNo, dbID, name, NULL, NULL, NULL, NULL,
	                  NULL, NULL, NULL, NULL, NULL, NULL);
	return true;
}	

/*************************************************************************
 *
 *    Function Name: DocListDBUpdateCategory
 *
 *    Purpose:       Update DocListDB and Doc file to new category
 *
 *    Parameters:    recordIndex: Index of record
 *                   newCategory: New category
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
void DocListDBUpdateCategory(UInt16 recordIndex, UInt16 newCategory)
{
	UInt16 cardNo;
	LocalID dbID;
	Char name[dmDBNameLength];
	UInt16 docAttr;
	DmOpenRef docR;
	
	// Get Doc info from DocDB
	DocListDBGetInfo(recordIndex, &cardNo, &dbID, name, NULL, NULL, NULL,
	                 NULL);
	                 
	// Open doc file
	docR = DmOpenDatabase(cardNo, dbID, dmModeReadWrite);
	
	// Get attributes of record 0
	DmRecordInfo(docR, 0, &docAttr, NULL, NULL);
	
	// Keep non-category parts of doc attributes
	docAttr &= ~dmRecAttrCategoryMask;
	
	// Ignore non-category parts of newCategory
	newCategory &= dmRecAttrCategoryMask;
	
	// Combine category and non-category attributes
	docAttr |= newCategory;
	
	// Set attributes of record 0
	DmSetRecordInfo(docR, 0, &docAttr, NULL);
	
	// Close Doc file
	DmCloseDatabase(docR);
	
	// Set DB record category
	DocListDBSetCategory(recordIndex);
}	

/*************************************************************************
 *
 *    Function Name: DocListDBQueryNextInCategory
 *
 *    Purpose:       Return handle to next record in specified category
 *
 *    Parameters:    In:
 *                   indexP:   Index of current record
 *                   category: Category to use
 *                   Out:
 *                   indexP:   Index of new record
 *
 *    Return Value:  Handle to next record in DocListDB
 *
 *
 *************************************************************************/
MemHandle DocListDBQueryNextInCategory (UInt16 *indexP,
                                        UInt16 category)
{
	return DmQueryNextInCategory(gDocListDBRef, indexP, category);
}

/*************************************************************************
 *
 *    Function Name: DocListDBSeekRecordInCategory
 *
 *    Purpose:       Return the index of the record nearest the offset
 *                   from the passed record index whose category matches
 *                   the passed category. (The offset parameter indicates
 *                   the number of records to move forward or backward.)
 *
 *    Parameters:    In:
 *                   indexP:    Index of current record
 *                   offset:    Offset of the passed record index
 *                   direction: dmSeekForward or dmSeekBackward
 *                   category:  Category to use
 *                   Out:
 *                   indexP:    Index of new record
 *
 *    Return Value:  Error if any
 *
 *
 *************************************************************************/
Err DocListDBSeekRecordInCategory (UInt16 *indexP, Int16 offset,
                                        Int16 direction, UInt16 category)
{
	// ####
	Err error;
	error = DmSeekRecordInCategory(gDocListDBRef, indexP, offset, direction,
	                              category);
	                              
	return error;
}
/*************************************************************************
 *
 *    Function Name: DocListDBCategoryGetName
 *
 *    Purpose:       Return the category name
 *
 *    Parameters:    In:
 *                   category: Category index
 *                   name:   Must point to Char array of dmCategoryLength
 *                   Out:
 *                   name:   Name of category indexed
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
void DocListDBCategoryGetName (UInt16 category, Char *name)
{
	CategoryGetName(gDocListDBRef, category, name);
}

/*************************************************************************
 *
 *    Function Name: DocListDBCategorySelect
 *
 *    Purpose:       Call CategorySelect
 *
 *    Parameters:    In:
 *                   frmP:      Form that contains the category popup list
 *                   ctlID:     ID of the popup trigger
 *                   lstID:     ID of the popup list
 *                   title:     Whether or not "All" should be in the list
 *                   categoryP: Index of current category
 *                   categoryName: Name of current category
 *
 *                   Out:
 *                   categoryP: Index of new category
 *                   categoryName: Name of new category
 *
 *    Return Value:  True if current category is renamed, deleted or merged
 *
 *
 *************************************************************************/
Boolean DocListDBCategorySelect(FormPtr frmP, UInt16 ctlID, UInt16 lstID,
                                Boolean title, UInt16 *categoryP,
                                Char *categoryName)
{
	Boolean categoryEdited;
	
	categoryEdited = CategorySelect(gDocListDBRef, frmP, ctlID, lstID,
	                                title, categoryP, categoryName,
	                                NumUneditableCategories, 0);
	
	return categoryEdited;
}
