/*************************************************************************
 *
 *    File Name: beam.c
 *
 *    Description: Handle sending databases using exchange manager
 *
 *    History:
 *       12 Apr 2000: Created; Joshua Colvin
 *       27 Apr 2000: Added BeamDoc, BeamCategory; Joshua Colvin
 *
 *************************************************************************
 * Copyright (C) 2000 Jason Campbell, Joshua Colvin, Jason Sherrill,
 *                    Ben Tobin
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, In., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 *************************************************************************/

#include <PalmOS.h>
#include "beam.h"
#include "doclistdb.h"

// Constants used when beaming Doc files
#define BEAM_INFO " (Doc File)"
#define BEAM_FILENAME "doc.pdb"

// Private function prototypes
static Err BeamCustomWrite (const void* dataP, UInt32* sizeP,
                            void* userDataP);

/*************************************************************************
 *
 *    Function Name: BeamCustomWrite
 *
 *    Purpose:       Callback function called by ExgDBWrite to send
 *                    data through the IR port.
 *
 *    Parameters:    dataP:     Data to send
 *                   sizeP:     Amount of data to send
 *                   userDataP: Pointer to socket structure
 *
 *    Return Value:  error code from ExgSend, if any
 *
 *
 *************************************************************************/
static Err BeamCustomWrite (const void* dataP, UInt32* sizeP,
                            void* userDataP)
{
	Err err;
	
	*sizeP = ExgSend(userDataP, dataP, *sizeP, &err);
	
	return err;
}

/*************************************************************************
 *
 *    Function Name: BeamDatabase
 *
 *    Purpose:       Beam entire database using the exchange manager
 *
 *    Parameters:    cardNumber:   Card number database resides on
 *                   dbID:         Local ID of database
 *                   nameP:        Pointer to database name string
 *                   descriptionP: Pointer to description string
 *
 *    Return Value:  errNone if no errors, or error from exchange manager
 *
 *
 *************************************************************************/
Err BeamDatabase(UInt16 cardNumber, LocalID dbID,
                 Char *nameP, Char *descriptionP)
{
	Err err;
	ExgSocketType sendingSocket;
	
	// Use default values for all members of sendingSocket
	MemSet(&sendingSocket, sizeof(sendingSocket), 0);
	
	// Set name and description of data to send through sendingSocket
	sendingSocket.name = nameP;
	sendingSocket.description = descriptionP;
	
	// Open connection using exchange manager
	err = ExgPut(&sendingSocket);
	if (err)
	{
		return err;
	}
	else // Connection was opened without any errors
	{
		// Send database using exchange manager
		err = ExgDBWrite(BeamCustomWrite, &sendingSocket, nameP,
		           dbID, cardNumber);
		
		// Close connection
		err = ExgDisconnect(&sendingSocket, err);
	}
	return err;
}

/*************************************************************************
 *
 *    Function Name: BeamDoc
 *
 *    Purpose:       Beam (transfer) Doc file using IR
 *
 *    Parameters:    docIndex: index ofDoc file to send in DocListDB
 *
 *    Return Value:  non-zero if error occurs
 *
 *
 *************************************************************************/
Err BeamDoc(UInt16 docIndex)
{
	Err err;
	UInt16 cardNumber;
	LocalID dbID;
	Char name[dmDBNameLength + sizeof(BEAM_INFO)];

	// Get info on Doc file
	DocListDBGetInfo(docIndex, &cardNumber, &dbID,
	                 name, NULL, NULL, NULL, NULL);

   StrCat(name, BEAM_INFO);
	err = BeamDatabase(cardNumber, dbID, BEAM_FILENAME, name);
	
	return err;
}

/*************************************************************************
 *
 *    Function Name: BeamCategory
 *
 *    Purpose:       Beam (transfer) Doc file(s) in category using IR
 *
 *    Parameters:    category: category of Doc files to transfer
 *
 *    Return Value:  non-zero if error occurs
 *
 *
 *************************************************************************/
Err BeamCategory(UInt16 category)
{
	Err err;
	UInt16 recordIndex;
	MemHandle recordH;
	
	// the var. recordH is just used to verify that we have a valid index
	recordH = NULL;
	
	// Initialize variables
	err = 0;
	// Start at top of list
	recordIndex = 0;
	
	// Get first record in category
	recordH = DocListDBQueryNextInCategory(&recordIndex, category);
	
	// Loop until all records in category are sent, or an error occurs
	while ( (NULL != recordH) && (0 == err) )
	{
		// Send Doc file
		err = BeamDoc(recordIndex);
		
		// Get next record in category
		DocListDBQueryNextInCategory(&recordIndex, category);
	}
	return err;
}
